package main

import (
	"context"
	"flag"
	"fmt"
	"log"
	"os"
	"path/filepath"

	"github.com/niveus/cloud-build-go/cloudbuild"
	"github.com/niveus/cloud-build-go/config"
)

func main() {
	// Define command-line flags
	projectID := flag.String("project", "", "Google Cloud Project ID")
	location := flag.String("location", "global", "Location for the connection (e.g., global, us-central1)")
	connectionName := flag.String("name", "", "Name for the connection")
	connectionType := flag.String("type", "GITHUB", "Type of connection (e.g., GITHUB, GITLAB)")
	action := flag.String("action", "create", "Action to perform: create, get, list, delete, create-gitlab")
	configPath := flag.String("config", "", "Path to config file (optional)")
	token := flag.String("token", "", "Access token for authentication (optional)")
	flag.Parse()

	// Project ID can come from config file or command line
	var projectIDValue, locationValue, connTypeValue string
	
	// Load config file if specified
	if *configPath != "" {
		cfg, err := config.LoadConfig(*configPath)
		if err != nil {
			log.Printf("Warning: Failed to load config file: %v", err)
		} else {
			log.Printf("Loaded configuration from %s (Project: %s, Location: %s)", 
				*configPath, cfg.ProjectID, cfg.Location)
			
			// Set defaults from config file
			projectIDValue = cfg.ProjectID
			locationValue = cfg.Location
			connTypeValue = cfg.DefaultConnType
			
			// Log the values that will be used
			log.Printf("Config values initially set to: Project=%s, Location=%s, Type=%s", 
				projectIDValue, locationValue, connTypeValue)
		}
	}
	
	// Command line arguments take precedence ONLY if they have non-default values
	if *projectID != "" {
		projectIDValue = *projectID
		log.Printf("Using command-line project ID: '%s'", projectIDValue)
	}
	
	if *location != "" && *location != "global" {
		locationValue = *location
		log.Printf("Using command-line location: '%s'", locationValue)
	}
	
	if *connectionType != "" && *connectionType != "GITHUB" {
		connTypeValue = *connectionType
		log.Printf("Using command-line connection type: '%s'", connTypeValue)
	}
	
	// Check for required values after potentially loading from config
	if projectIDValue == "" {
		log.Fatal("project ID is required (provide via --project flag or config file)")
	}

	// Additional validation based on the action
	if (*action == "create" || *action == "get" || *action == "delete" || *action == "listrepos" || *action == "inspect") && *connectionName == "" {
		log.Fatal("name flag is required for create, get, delete, listrepos, and inspect actions")
	}

	ctx := context.Background()

	// Create a Cloud Build client using our package
	var client *cloudbuild.Client
	var err error
	
	// Authentication approach
	// If a token is provided via command line, use that
	// Otherwise, rely on Application Default Credentials
	if *token != "" {
		log.Printf("Using provided access token for authentication")
		client, err = cloudbuild.NewClientWithToken(ctx, *token)
	} else {
		// Use Application Default Credentials
		log.Printf("Using Application Default Credentials for authentication")
		client, err = cloudbuild.NewClient(ctx)
	}

	if err != nil {
		log.Fatalf("Failed to create Cloud Build client: %v", err)
	}

	// Perform the requested action
	switch *action {
	case "create":
		err = client.CreateConnection(projectIDValue, locationValue, *connectionName, connTypeValue)
		if err != nil {
			log.Fatalf("Failed to create connection: %v", err)
		}
		fmt.Printf("Successfully created connection: %s\n", fmt.Sprintf("projects/%s/locations/%s/connections/%s", projectIDValue, locationValue, *connectionName))
		
	case "create-gitlab":
		// Check if we have a configuration for GitLab
		var connConfig *config.ConnectionConfig
		if *configPath != "" {
			cfg, err := config.LoadConfig(*configPath)
			if err != nil {
				log.Fatalf("Failed to load config file: %v", err)
			}
			
			// Find the connection config for the given name
			for _, conn := range cfg.Connections {
				if conn.Name == *connectionName || *connectionName == "" && len(cfg.Connections) == 1 {
					connConfig = &conn
					break
				}
			}
			
			if connConfig == nil {
				log.Fatalf("Connection configuration not found for '%s'", *connectionName)
			}
			
			if connConfig.GitlabConfig == nil {
				log.Fatalf("GitLab configuration not found for connection '%s'", connConfig.Name)
			}
			
			// For GitLab connections, we'll use the values directly from the config file
			// This is the most reliable way to ensure we're using the right values
			projectIDValue = cfg.ProjectID
			locationValue = cfg.Location
			
			log.Printf("For GitLab connection, using config values: Project='%s', Location='%s'", 
				projectIDValue, locationValue)
			
			fmt.Printf("Creating GitLab connection '%s' in project '%s' (location: '%s')\n", 
				connConfig.Name, projectIDValue, locationValue)
			
			// Create the GitLab connection using the Cloud Build SDK
			// This uses the official SDK methods, not manual URL construction
			err = client.CreateGitLabConnection(
				projectIDValue, 
				locationValue, 
				connConfig.Name, 
				connConfig.GitlabConfig,
				connConfig.Annotations,
			)
			if err != nil {
				log.Fatalf("Failed to create GitLab connection: %v", err)
			}
			
			fmt.Printf("Successfully created GitLab connection: %s\n", 
				fmt.Sprintf("projects/%s/locations/%s/connections/%s", projectIDValue, locationValue, connConfig.Name))
		} else {
			log.Fatal("Config file is required for creating GitLab connections")
		}

		// Save example config if it doesn't exist
		if *configPath == "" {
			defaultConfigPath := filepath.Join("config", "config.json")
			_, err := os.Stat(defaultConfigPath)
			if os.IsNotExist(err) {
				cfg := &config.Config{
					ProjectID:       projectIDValue,
					Location:        locationValue,
					DefaultConnType: connTypeValue,
				}
				err = config.SaveConfig(cfg, defaultConfigPath)
				if err != nil {
					log.Printf("Note: Failed to save example config file: %v", err)
				} else {
					log.Printf("Note: Created example config file at %s", defaultConfigPath)
				}
			}
		}

	case "get":
		connection, err := client.GetConnection(projectIDValue, locationValue, *connectionName)
		if err != nil {
			log.Fatalf("Failed to get connection: %v", err)
		}
		fmt.Printf("Connection details:\nName: %s\nDisabled: %t\n", connection.Name, connection.Disabled)
		
		// Show type-specific details
		if connection.GithubConfig != nil {
			fmt.Printf("Type: GitHub\n")
			if connection.GithubConfig.AppInstallationId > 0 {
				fmt.Printf("App Installation ID: %d\n", connection.GithubConfig.AppInstallationId)
			}
		} else if connection.GitlabConfig != nil {
			fmt.Printf("Type: GitLab\n")
			fmt.Printf("Host URI: %s\n", connection.GitlabConfig.HostUri)
		}

	case "list":
		connections, err := client.ListConnections(projectIDValue, locationValue)
		if err != nil {
			log.Fatalf("Failed to list connections: %v", err)
		}
		fmt.Printf("Found %d connections in project %s (location: %s):\n", 
			len(connections), projectIDValue, locationValue)
		for i, conn := range connections {
			// Determine connection type based on config
			connType := "Unknown"
			if conn.GithubConfig != nil {
				connType = "GitHub"
			} else if conn.GitlabConfig != nil {
				connType = "GitLab"
			}
			fmt.Printf("%d. %s (Type: %s)\n", i+1, conn.Name, connType)
		}

	case "delete":
		err := client.DeleteConnection(projectIDValue, locationValue, *connectionName)
		if err != nil {
			log.Fatalf("Failed to delete connection: %v", err)
		}
		fmt.Printf("Successfully deleted connection: %s\n", *connectionName)

	case "listrepos":
		repositories, err := client.ListLinkableRepositories(projectIDValue, locationValue, *connectionName)
		if err != nil {
			log.Fatalf("Failed to list linkable repositories: %v", err)
		}
		fmt.Printf("Found %d linkable repositories for connection %s:\n", 
			len(repositories), *connectionName)
		for i, repo := range repositories {
			fmt.Printf("%d. %s\n", i+1, repo.Name)
			if repo.RemoteUri != "" {
				fmt.Printf("   URI: %s\n", repo.RemoteUri)
			}
			// Add a blank line between repositories for better readability
			if i < len(repositories)-1 {
				fmt.Println()
			}
		}
	
	case "inspect":
		// This action provides detailed inspection of a connection's configuration and status
		fmt.Printf("Inspecting connection %s in project %s (location: %s)...\n\n", 
			*connectionName, projectIDValue, locationValue)
		err := client.InspectConnection(projectIDValue, locationValue, *connectionName)
		if err != nil {
			log.Fatalf("Failed to inspect connection: %v", err)
		}
		
	default:
		log.Fatalf("Unknown action: %s. Supported actions are: create, get, list, delete, listrepos, inspect", *action)
	}
}
