/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/
import { Test, TestingModule } from '@nestjs/testing';
import { MicroserviceHealthService } from './microservice.service';
import { ModuleRef } from '@nestjs/core';
import { of, throwError } from 'rxjs';
import { MICROSERVICE_CONFIGS } from './microservices.config';

describe('MicroserviceHealthService', () => {
  let service: MicroserviceHealthService;
  const mockClientProxy = {
    send: jest.fn(),
  };

  const mockModuleRef = {
    get: jest.fn(),
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [
        MicroserviceHealthService,
        { provide: ModuleRef, useValue: mockModuleRef },
      ],
    }).compile();

    service = module.get<MicroserviceHealthService>(MicroserviceHealthService);
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  describe('isServiceAvailable', () => {
    it('should return true when microservice responds with pong',
        async () => {
          mockClientProxy.send.mockReturnValue(of('pong'));
          mockModuleRef.get.mockReturnValue(mockClientProxy);

          const result = await service.isServiceAvailable('AUTH_SERVICE');
          expect(result).toBe(true);
          expect(mockModuleRef.get).toHaveBeenCalledWith('AUTH_SERVICE', {
            strict: false,
          });
          expect(mockClientProxy.send).toHaveBeenCalledWith({ cmd: 'ping' }, {});
        });

    it('should return false when microservice responds with non-pong',
        async () => {
          mockClientProxy.send.mockReturnValue(of('not-pong'));
          mockModuleRef.get.mockReturnValue(mockClientProxy);

          const result =
      await service.isServiceAvailable('AUTH_SERVICE');
          expect(result).toBe(false);
        });

    it(
        'should return false and log error when microservice throws an error',
        async () => {
          mockClientProxy.send.mockReturnValue(
              throwError(() => new Error('Connection failed')),
          );
          mockModuleRef.get.mockReturnValue(mockClientProxy);

          const result =
      await service.isServiceAvailable('AUTH_SERVICE');
          expect(result).toBe(false);
        });
  });

  describe('checkMultipleServices', () => {
    it('should return status for each service token', async () => {
      const tokens = ['AUTH_SERVICE', 'USER_SERVICE'];

      jest
          .spyOn(service, 'isServiceAvailable')
          .mockImplementation(async (token) => token === 'AUTH_SERVICE');

      const result = await service.checkMultipleServices(tokens);

      expect(result).toEqual({
        AUTH_SERVICE: true,
        USER_SERVICE: false,
      });
    });
  });
});

describe('MICROSERVICE_CONFIGS', () => {
  it('should be defined', () => {
    expect(MICROSERVICE_CONFIGS).toBeDefined();
    expect(Array.isArray(MICROSERVICE_CONFIGS)).toBe(true);
  });

  it('should contain CASBIN_SERVICE config', () => {
    const casbin = MICROSERVICE_CONFIGS.find(
        (service) => service.name === 'CASBIN_SERVICE',
    );
    expect(casbin).toBeDefined();
    expect(casbin).toEqual({
      name: 'CASBIN_SERVICE',
      host: 'localhost',
      port: 8001,
    });
  });

  it('should contain INTEGRATION_SERVICE config', () => {
    const integration = MICROSERVICE_CONFIGS.find(
        (service) => service.name === 'INTEGRATION_SERVICE',
    );
    expect(integration).toBeDefined();
    expect(integration).toEqual({
      name: 'INTEGRATION_SERVICE',
      host: 'localhost',
      port: 8002,
    });
  });

  it('should contain exactly two configs', () => {
    expect(MICROSERVICE_CONFIGS).toHaveLength(2);
  });
});
