/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import { Injectable } from '@nestjs/common';
import { PrismaService } from '../prisma/prisma.service';
import { fuzzySearch, Logger, handlePrismaError } from 'nest-common-utilities';
import {
  GetHospitalsRequestDto,
} from './dto/get-hospitals/get-hospital-list-request.dto';
import {
  GetHospitalsResponseDto
} from './dto/get-hospitals/get-hospital-list-response.dto';
import { Decimal } from '@prisma/client/runtime/library';

/**
 * Service to encapsulate hospital domain logic such as user search.
 */
@Injectable()
export class HospitalService {
  private logger = new Logger(HospitalService.name);
  /**
   * Constructs the HospitalService.
   *
   * @param prisma - The Prisma service for database interactions.
   */
  constructor(private readonly prisma: PrismaService) { }

  // Public transform method
  /**
   * Return plain hospital data.
   * @param rawHospitals
   * @returns
   */
  transformHospitalData(rawHospitals: any[]) {
    return rawHospitals.map((h) => ({
      id: h.hospital_id,
      hospitalName: h.name,
      hospitalNo: h.hospital_no,
      rohiniId: h.rohini_id,
      hfrId: h.hfr_id,
      registrationNo: h.registration_no,
      hospitalRegNo: h.registration_no,
      registrationAuthority: h.registration_authority,
      website: h.website,
      isHospitalRegistered: h.is_hospital_registered,
      registeredFrom: h.registered_from,
      registeredTill: h.registered_till,
      type: h.hospital_type_master?.name ?? null,
      propreitorName: h.propreitor_name,
      panNo: h.pan_no,
      address: h.address,
      locality: h.locality,
      state: h.state_master?.name ?? null,
      city: h.city_master?.name ?? null,
      pincode: h.pincode_master?.name ?? null,
      contactNo: h.contact,
      emailId: h.email_id,
      status: h.status,
      remarks: h.remarks,
      stateId: h.state_id,
      cityId: h.city_id,
      pincodeId: h.pincode_id,
      latitude: new Decimal(h.latitude).toNumber(),
      longitude: new Decimal(h.longitude).toNumber(),
      isEPDEnabled: h.is_epd_enabled,
      isNonNetwork: h.is_non_network
    }));
  }

  /**
   * Calculates pagination parameters based on the provided page and limit.
   *
   * @param page - Page number (defaults to 1 if not provided or invalid).
   * @param limit - Number of records per page (defaults to 10 if invalid).
   * @returns An object containing `limit` and `offset` values for pagination.
   */
  async pagination(
    page?: number,
    limit?: number,
  ): Promise<{ limit: number; offset: number }> {
    const safeLimit = limit && limit > 0 ? limit : 10;
    const safePage = page && page > 0 ? page : 1;
    const offset = (safePage - 1) * safeLimit;
    return { limit: safeLimit, offset };
  }

  /**
   * Fetches the hospital list by hospital ID.
   *
   * @param dto - The data transfer object containing hospitalId and pagination.
   * @param GetHospitalsRequestDto
   * @returns A promise that resolves to the hospital list and total count.
   */
  async getHospitalsByName(
    GetHospitalsRequestDto: GetHospitalsRequestDto,
  ): Promise<GetHospitalsResponseDto> {
    const { hospitalName, stateId, cityId, pincodeId, status, page, limit } =
      GetHospitalsRequestDto;
    const { limit: take, offset: skip } = await this.pagination(page, limit);

    const where: Record<string, any> = {};
    if (status) where.status = status;
    if (stateId) where.state_id = stateId;
    if (cityId) where.city_id = cityId;
    if (pincodeId) where.pincode_id = pincodeId;

    try {
      const data = await fuzzySearch<any>(this.prisma, 'hospitals', {
        searchFields: ['name'],
        searchTerm: hospitalName || '',
        limit: take,
        offset: skip,
        similarityThreshold: 0.1,
        where,
        include: {
          pincode_master: true,
          state_master: true,
          city_master: true,
          hospital_type_master: true
        },
        primaryKey: 'hospital_id',
      });
      return {
        totalCount: data.total,
        list: this.transformHospitalData(data.data),
      };
    } catch (error) {
      this.logger.error('Error fetching hospitals:', error);
      handlePrismaError(error);
    }
  }

  /**
   * Fetched the hospital lists by using hospital name.
   *
   * @param {GetHospitalsRequestDto} dto - The data transfer object
   *   containing the user information to be created.
   * @returns {Promise<GetHospitalsResponseDto>} A promise that resolves
   *   to the created user’s data.
   */
  async getHospitalByHospitalId(
    dto: GetHospitalsRequestDto,
  ): Promise<GetHospitalsResponseDto> {
    const { hospitalId, page, limit } = dto;
    const { limit: take, offset: skip } = await this.pagination(page, limit);
    const [hospitalsData, total] = await Promise.all([
      this.prisma.hospitals.findMany({
        where: {
          OR: [
            { hfr_id: hospitalId },
            { hospital_no: hospitalId },
          ],
        },
        skip,
        take,
        include: {
          pincode_master: true,
          state_master: true,
          city_master: true,
          hospital_type_master: true
        },
      }),
      this.prisma.hospitals.count({
        where: { hfr_id: hospitalId },
      }),
    ]);
    return {
      totalCount: total,
      list: this.transformHospitalData(hospitalsData),
    };
  }

  /**
   * Fetches the hospital list by hospital Email ID.
   *
   * @param dto - The data transfer object containing emailId and pagination.
   * @returns A promise that resolves to the hospital list and total count.
   */
  async getHospitalByEmailId(
    dto: GetHospitalsRequestDto,
  ): Promise<GetHospitalsResponseDto> {
    const { emailId, page, limit } = dto;
    const { limit: take, offset: skip } = await this.pagination(page, limit);

    const [hospitalsData, total] = await Promise.all([
      this.prisma.hospitals.findMany({
        where: { email_id: emailId },
        skip,
        take,
        include: {
          pincode_master: true,
          state_master: true,
          city_master: true,
          hospital_type_master: true
        },
      }),
      this.prisma.hospitals.count({
        where: { email_id: emailId },
      }),
    ]);

    return {
      totalCount: total,
      list: this.transformHospitalData(hospitalsData),
    };
  }
}
