/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/
import { Test, TestingModule } from '@nestjs/testing';
import { HospitalService } from './hospital.service';
import { PrismaService } from '../prisma/prisma.service';
import { Decimal } from '@prisma/client/runtime/library';

let service: HospitalService;
const mockPrisma = {
  hospitals: {
    findMany: jest.fn(),
    count: jest.fn(),
  },
};
jest.mock('nest-common-utilities', () => {
  return {
    fuzzySearch: jest.fn(),
    handlePrismaError: jest.fn(),
    Logger: jest.fn().mockImplementation(() => ({
      log: jest.fn(),
      error: jest.fn(),
    })),
  };
});
import { fuzzySearch } from 'nest-common-utilities';

beforeEach(async () => {
  const module: TestingModule = await Test.createTestingModule({
    providers: [
      HospitalService,
      { provide: PrismaService, useValue: mockPrisma },
    ],
  }).compile();
  service = module.get<HospitalService>(HospitalService);
});
afterEach(() => {
  jest.clearAllMocks();
});

describe('HospitalService', () => {
  describe('pagination', () => {
    it(
        'should return default pagination values if invalid input provided',
        async () => {
          const result = await service.pagination(undefined, undefined);
          expect(result).toEqual({ limit: 10, offset: 0 });
        });
    it('should return correct pagination values', async () => {
      const result = await service.pagination(2, 5);
      expect(result).toEqual({ limit: 5, offset: 5 });
    });
  });
  describe('getHospitalByHospitalId', () => {
    it('should return hospital data by hospitalId', async () => {
      const mockDto = {
        hospitalId: 'HOSP123',
        page: 1,
        limit: 10,
        _customValidationPlaceholder: true,
      };
      const mockHospitals = [
        {
          hospital_id: '1', hospital_no: '001', rohini_id: 'ROH123',
          hfr_id: 'HOSP123', name: 'Test Hospital',
          address: '123 Street', locality: 'Town',
          pincode_id: 101, email_id: 'test@hospital.com',
          website: 'https://testhospital.com',
          is_hospital_registered: true, registration_no: 'REG001',
          registration_authority: 'Govt', registered_from: '2021-01-01',
          registered_till: '2030-01-01', latitude: new Decimal(12.9716),
          longitude: new Decimal(77.5946), contact_hash: '1234567890',
          is_epd_enabled: false, is_non_network: false, status: 'active',
          remarks: 'Good', pincode_master: { id: 101, name: '560001' },
        },
      ];
      mockPrisma.hospitals.findMany.mockResolvedValue(mockHospitals);
      mockPrisma.hospitals.count.mockResolvedValue(1);
      const result = await service.getHospitalByHospitalId(mockDto);
      expect(mockPrisma.hospitals.findMany).toHaveBeenCalledWith(
        expect.objectContaining({
          where: {
            OR: [
              { hfr_id: 'HOSP123' },
              { hospital_no: 'HOSP123' },
            ],
          },
          skip: 0,
          take: 10,
          include: {
            pincode_master: true,
            state_master: true,
            city_master: true,
            hospital_type_master: true,
          },
        }),
      );
      expect(result.totalCount).toBe(1);
      expect(result.list[0].hospitalName).toBe('Test Hospital');
    });
  });
  describe('getHospitalByEmailId', () => {
    it('should return hospital data by emailId', async () => {
      const mockDto = { emailId: 'test@hospital.com', page: 1, limit: 10,
        _customValidationPlaceholder: true,
      };
      const mockHospitals = [
        {
          id: '2', hospital_no: '002', rohini_id: 'ROH124', hfr_id: 'HOSP124',
          name: 'Email Hospital', address: '456 Avenue', locality: 'City',
          pincode_id: 102,
          email_id: 'test@hospital.com', website: 'https://emailhospital.com',
          is_hospital_registered: true, registration_no: 'REG002',
          registration_authority: 'Govt', registered_from: '2022-01-01',
          registered_till: '2031-01-01', latitude: new Decimal(11.1111),
          longitude: new Decimal(75.5555), contact_hash: '0987654321',
          is_epd_enabled: true, is_non_network: false, status: 'active',
          remarks: 'Verified', pincode_master: { id: 102, name: '560002' },
        },
      ];
      mockPrisma.hospitals.findMany.mockResolvedValue(mockHospitals);
      mockPrisma.hospitals.count.mockResolvedValue(1);
      const result = await service.getHospitalByEmailId(mockDto);
      expect(result.totalCount).toBe(1);
      expect(result.list[0].emailId).toBe('test@hospital.com');
    });
  });
  describe('transformHospitalData', () => {
    it('should transform raw hospital data into expected format', () => {
      const input = [
        {
          id: '123', hospital_no: '001', rohini_id: 'R001', hfr_id: 'H001',
          name: 'Test Hospital', address: '123 Street', locality: 'Area',
          pincode_id: 200, pincode_master: { id: 200, name: '600001' },
          email_id: 'test@example.com', website: 'http://example.com',
          is_hospital_registered: true, registration_no: 'REG001',
          registration_authority: 'Health Dept', registered_from: '2020-01-01',
          registered_till: '2030-01-01', latitude: new Decimal(10.1234),
          longitude: new Decimal(20.5678), contact_hash: '123456',
          is_epd_enabled: true, is_non_network: false,
          status: 'active', remarks: 'Verified',
          state_master: {id: '560001',name: 'Maharashtra'},
          city_master: {id: '560001',name: 'Mumbai'}
        },
      ];
      const result = service.transformHospitalData(input);
      expect(result[0].pincode).toBe('600001');
    });
  });
});

describe('getHospitalsByName', () => {
  it('should return hospital data based on hospital name', async () => {
    const mockDto = {
      hospitalName: 'Sunshine Hospital Delhi',
      stateId: '003c2ad6-cd04-45f9-8c2b-ee0d009e3a06',
      cityId: '697e3c42-d7b6-4994-906f-ae1490e7155a',
      pincodeId: '6044da1b-bbd7-442f-9bd7-c4a717e40c2c',
      status: ['ACTIVE'],
      page: 1,
      limit: 2,
      _customValidationPlaceholder: true,
    };

    const mockFuzzyResult = {
      total: 1,
      data: [
        {
          id: '1',
          hospital_no: '001',
          rohini_id: 'ROH001',
          hfr_id: 'HFR001',
          name: 'Sunshine Hospital Delhi',
          address: 'MG Road',
          locality: 'Central',
          pincode_id: '560001',
          email_id: 'apollo@hospital.com',
          website: 'https://apollo.com',
          is_hospital_registered: true,
          registration_no: 'REG123',
          registration_authority: 'Govt',
          registered_from: '2020-01-01',
          registered_till: '2030-01-01',
          latitude: new Decimal(12.9716),
          longitude: new Decimal(77.5946),
          contact_hash: '9876543210',
          is_epd_enabled: true,
          is_non_network: false,
          status: 'ACTIVE',
          remarks: '',
          state_id: '003c2ad6-cd04-45f9-8c2b-ee0d009e3a06',
          city_id: '697e3c42-d7b6-4994-906f-ae1490e7155a',
          pincode_master: {
            id: '560001',
            name: '560001',
          },
          hospital_type_master: {
            id: '560001',
            name: 'Type 1',
          },
        },
      ],
    };

    // Mock fuzzySearch to return mockFuzzyResult
    (fuzzySearch as jest.Mock).mockResolvedValue(mockFuzzyResult);

    const result = await service.getHospitalsByName(mockDto);

    expect(result.totalCount).toBe(1);
    expect(result.list[0].hospitalName).toBe('Sunshine Hospital Delhi');
  });

  it(
      'should handle missing hospitalName and fallback to empty string',
      async () => {
        const mockDto = {
          stateId: '003c2ad6-cd04-45f9-8c2b-ee0d009e3a06',
          cityId: '697e3c42-d7b6-4994-906f-ae1490e7155a',
          pincodeId: '6044da1b-bbd7-442f-9bd7-c4a717e40c2c',
          status: ['ACTIVE'],
          page: 1,
          limit: 2,
          _customValidationPlaceholder: true,
          // hospitalName intentionally omitted
        };

        const mockFuzzyResult = {
          total: 0,
          data: [],
        };

        (fuzzySearch as jest.Mock).mockResolvedValue(mockFuzzyResult);

        const result = await service.getHospitalsByName(mockDto);

        expect(fuzzySearch).toHaveBeenCalledWith(
            expect.anything(),
            'hospitals',
            expect.objectContaining({
              searchTerm: '', // ensures fallback works
            }),
        );

        expect(result.totalCount).toBe(0);
        expect(result.list).toEqual([]);
      });

      it('should handle errors thrown by fuzzySearch', async () => {
        const error = new Error('Simulated fuzzySearch failure');
        (fuzzySearch as jest.Mock).mockRejectedValue(error);
      
        const dto = {
          hospitalName: 'Test',
          page: 1,
          limit: 10,
        } as any;
      
        const loggerErrorSpy = jest.spyOn(service['logger'], 'error').mockImplementation(() => {});
      
        await expect(service.getHospitalsByName(dto)).resolves.toBeUndefined();
      
        expect(loggerErrorSpy).toHaveBeenCalledWith('Error fetching hospitals:', error);
        loggerErrorSpy.mockRestore();
      });
});
