/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import { Test, TestingModule } from '@nestjs/testing';
import { DoctorController } from './doctor.controller';
import { DoctorService } from './doctor.service';
import { ListDoctorsRequestDto } from './dto/list-doctors-request.dto';

import { PaginatedDoctorsResponseDto } from './dto/paginated-doctors-response.dto';

describe('DoctorController', () => {
  let controller: DoctorController;
  let service: DoctorService;

  // Base mock data
  const baseDoctor = {
    id: 'e82f244c-91f7-4609-ad5d-fff5fac36f8f',
    doctorName: 'Dr. Ashwin Shetty',
    doctorRegNo: 'HR100926',
    specialityId: '0128716e-b5db-4fab-ba39-75439b97cdb1',
    speciality: 'Neurologist',
    qualificationIds: [],
    contactNo: '9876543210',
    hprCode: 'ADCVB987GH',
    emailId: 'ashwin.shetty@healthcare.com',
    panNo: 'CHPI789323',
    address: 'Marol 5th main',
    pincodeId: '102c4cf7-f0ed-4bca-a100-ee7857c3c772',
    pincode: '400011',
    latitude: '14.910813',
    longitude: '79.910813',
  };

  const baseDoctorWithQualification = {
    id: '41fb6feb-b18c-4d4e-a37b-5fb690b785a3',
    doctorName: 'Dr. Ashwin Shetty',
    doctorRegNo: 'HR1009267',
    specialityId: '0128716e-b5db-4fab-ba39-75439b97cdb1',
    speciality: 'Neurologist',
    qualificationIds: [
      {
        id: '010d7980-39a3-4ffc-b778-857bdf56166c',
        qualification: 'BDS',
      },
    ],
    contactNo: '9876543210',
    hprCode: 'ADCVB987GH',
    emailId: 'ashwin.shetty@healthcare.com',
    panNo: 'CHPI789323',
    address: 'Marol 5th main',
    pincodeId: '102c4cf7-f0ed-4bca-a100-ee7857c3c772',
    pincode: '400011',
    latitude: '14.910813',
    longitude: '79.910813',
  };

  // Mock responses
  const mockPrismaResponse = {
    data: [
      {
        id: 'e82f244c-91f7-4609-ad5d-fff5fac36f8f',
        name: 'Dr. Ashwin Shetty',
        registration_no: 'HR100926',
        speciality_id: '0128716e-b5db-4fab-ba39-75439b97cdb1',
        doctor_speciality_master: {
          id: '0128716e-b5db-4fab-ba39-75439b97cdb1',
          name: 'Neurologist',
        },
        qualifications: [],
        doctor_contact: '9876543210',
        hpr_code: 'ADCVB987GH',
        doctor_email_id: 'ashwin.shetty@healthcare.com',
        pan_no: 'CHPI789323',
        address: 'Marol 5th main',
        pincode_master: {
          id: '102c4cf7-f0ed-4bca-a100-ee7857c3c772',
          name: '400011',
        },
        latitude: 14.910813,
        longitude: 79.910813,
        created_by: 'e82f244c-91f7-4609-ad5d-fff5fac36f8f',
      },
      {
        id: '41fb6feb-b18c-4d4e-a37b-5fb690b785a3',
        name: 'Dr. Ashwin Shetty',
        registration_no: 'HR1009267',
        speciality_id: '0128716e-b5db-4fab-ba39-75439b97cdb1',
        doctor_speciality_master: {
          id: '0128716e-b5db-4fab-ba39-75439b97cdb1',
          name: 'Neurologist',
        },
        qualifications: [
          {
            id: '010d7980-39a3-4ffc-b778-857bdf56166c',
            qualification: {
              id: '010d7980-39a3-4ffc-b778-857bdf56166c',
              name: 'BDS',
            },
          },
        ],
        doctor_contact: '9876543210',
        hpr_code: 'ADCVB987GH',
        doctor_email_id: 'ashwin.shetty@healthcare.com',
        pan_no: 'CHPI789323',
        address: 'Marol 5th main',
        pincode_master: {
          id: '102c4cf7-f0ed-4bca-a100-ee7857c3c772',
          name: '400011',
        },
        latitude: 14.910813,
        longitude: 79.910813,
        created_by: '41fb6feb-b18c-4d4e-a37b-5fb690b785a3',
      },
    ],
    totalCount: 2,
  };

  const mockResponse: PaginatedDoctorsResponseDto = {
    data: [baseDoctor, baseDoctorWithQualification],
    totalCount: 2,
  };

  const mockEmptyResponse: PaginatedDoctorsResponseDto = {
    data: [],
    totalCount: 0,
  };

  const mockSingleDoctorResponse: PaginatedDoctorsResponseDto = {
    data: [baseDoctor],
    totalCount: 1,
  };

  // Base request for tests
  const baseRequest: ListDoctorsRequestDto = {
    doctorName: 'Dr. Ashwin Shetty',
    doctorRegNo: 'HR100926',
    emailId: 'ashwin.shetty@healthcare.com',
    mobileNo: '9876543210',
    page: 1,
    limit: 10,
  };

  // Test request data
  const testRequests = {
    base: baseRequest,
    empty: { doctorName: 'Non-existent Doctor' },
    nameOnly: { doctorName: 'Dr. Ashwin Shetty' },
    regNo: { doctorName: '', doctorRegNo: 'HR100926' },
    email: { doctorName: '', emailId: 'ashwin.shetty@healthcare.com' },
    pagination: { doctorName: '', page: 2, limit: 5 },
    mobile: { doctorName: '', mobileNo: '9876543210' },
    combined: { doctorName: 'Dr. Ashwin Shetty', doctorRegNo: 'HR100926', emailId: 'ashwin.shetty@healthcare.com', mobileNo: '9876543210' },
    defaultPagination: { doctorName: 'Dr. Ashwin Shetty' },
    edgeCase: { doctorName: '', page: 1000, limit: 100 },
    emptyFilters: { doctorName: '', doctorRegNo: '', emailId: '', mobileNo: '' },
    whitespace: {
      doctorName: '   Dr. Ashwin Shetty   ',
      doctorRegNo: '  HR100926  ',
      emailId: '  ashwin.shetty@healthcare.com  ',
      mobileNo: '  9876543210  ',
    },
    specialChars: {
      doctorName: 'Dr. Ashwin Shetty!@#$',
      doctorRegNo: 'HR100926!@#$',
      emailId: 'ashwin.shetty@healthcare.com!@#$',
      mobileNo: '9876543210!@#$',
    },
    caseInsensitive: {
      doctorName: 'dr. ashwin shetty',
      doctorRegNo: 'hr100926',
      emailId: 'ashwin.shetty@healthcare.com',
      mobileNo: '9876543210',
    },
  };

  // Helper function to create test cases
  const createTestRequest = (overrides: Partial<ListDoctorsRequestDto>): ListDoctorsRequestDto => ({
    ...baseRequest,
    ...overrides,
  });

  const createTest = (name: string, request: ListDoctorsRequestDto, response: PaginatedDoctorsResponseDto) => {
    it(name, async () => {
      jest.spyOn(service, 'getDoctors').mockResolvedValueOnce(response);
      const result = await controller.getDoctors(request);
      expect(result).toEqual(response);
      expect(service.getDoctors).toHaveBeenCalledWith(request);
    });
  };

  beforeEach(async () => {
    const module: TestingModule = await Test.createTestingModule({
      controllers: [DoctorController],
      providers: [
        {
          provide: DoctorService,
          useValue: {
            getDoctors: jest.fn().mockImplementation(async (request) => {
              const [doctors, totalCount] = await Promise.all([
                Promise.resolve(mockPrismaResponse.data),
                Promise.resolve(mockPrismaResponse.totalCount),
              ]);

              const transformedDoctors = doctors.map((doctor) => ({
                id: doctor.id,
                doctorName: doctor.name,
                doctorRegNo: doctor.registration_no,
                specialityId: doctor.speciality_id,
                speciality: doctor.doctor_speciality_master?.name,
                qualificationIds: doctor.qualifications?.map((qualification) => ({
                  id: qualification.id,
                  qualification: qualification.qualification.name,
                })) ?? [],
                contactNo: doctor.doctor_contact,
                hprCode: doctor.hpr_code,
                emailId: doctor.doctor_email_id,
                panNo: doctor.pan_no?.toString(),
                address: doctor.address,
                pincodeId: doctor.pincode_master?.id,
                pincode: doctor.pincode_master?.name,
                latitude: doctor.latitude,
                longitude: doctor.longitude,
              }));

              return {
                data: transformedDoctors,
                totalCount,
              };
            }),
          },
        },
      ],
    }).compile();

    controller = module.get<DoctorController>(DoctorController);
    service = module.get<DoctorService>(DoctorService);
  });

  it('should be defined', () => {
    expect(controller).toBeDefined();
  });

  describe('getDoctors', () => {
    createTest('should return paginated doctors with all filters', createTestRequest(testRequests.base), mockResponse);
    createTest('should return empty list when no doctors match', createTestRequest(testRequests.empty), mockEmptyResponse);
    createTest('should return single doctor when filtering by name', createTestRequest(testRequests.nameOnly), mockSingleDoctorResponse);
    createTest('should return doctors with registration number filter', createTestRequest(testRequests.regNo), mockResponse);
    createTest('should return doctors with email filter', createTestRequest(testRequests.email), mockResponse);
    createTest('should handle pagination parameters', createTestRequest(testRequests.pagination), mockResponse);
    createTest('should handle mobile number filter', createTestRequest(testRequests.mobile), mockResponse);
    createTest('should handle multiple filters combined', createTestRequest(testRequests.combined), mockResponse);
    createTest('should handle default pagination values', createTestRequest(testRequests.defaultPagination), mockResponse);
    createTest('should handle edge case pagination values', createTestRequest(testRequests.edgeCase), mockResponse);
    createTest('should handle empty string filters', createTestRequest(testRequests.emptyFilters), mockResponse);
    createTest('should handle whitespace in filters', createTestRequest(testRequests.whitespace), mockResponse);
    createTest('should handle special characters in filters', createTestRequest(testRequests.specialChars), mockResponse);
    createTest('should handle case-insensitive filters', createTestRequest(testRequests.caseInsensitive), mockResponse);
  });
});
