/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import { Controller, Post, Body } from '@nestjs/common';
import { DoctorService } from './doctor.service';
import { CreateDoctorDto } from './dto/create-doctor.dto';
import { DoctorCreateResponseDto } from './dto/doctor-dto';
import {
  ApiTags,
  ApiOperation,
  ApiCreatedResponse,
  ApiResponse,
} from '@nestjs/swagger';
import { Logger } from 'nest-common-utilities';
import { UpdateDoctorRequestDto } from './dto/update-doctor-request.dto';
import { UpdateDoctorResponseDto } from './dto/update-doctor-response.dto';
import { ListDoctorsRequestDto } from './dto/list-doctors-request.dto';
import {
  PaginatedDoctorsResponseDto,
} from './dto/paginated-doctors-response.dto';


/**
 * Controller to manage doctor-related API endpoints.
 */

/**
 *
 */
@ApiTags('doctors')
@Controller('doctors')
export class DoctorController {
  private logger = new Logger(DoctorService.name);

  /**
   * Initializes the DoctorsController.
   *
   * @param doctorsService - Service for handling doctors business logic.
   */

  /**
   * Doctor service params
   *
   * @param doctorService
   */
  constructor(private readonly doctorService: DoctorService) {}

  /**
   * Handles the HTTP POST request to create doctor's details.
   * Delegates the logic to the doctorService. Returns the created doctor
   * information in the response.
   *
   *
   * @param createDoctorDto - The DTO containing doctor update data.
   * @returns A promise that resolves to the created doctor response DTO.
   */
  @Post('create')
  @ApiOperation({ summary: 'Create a new doctor' })
  @ApiCreatedResponse({
    description: 'Doctor created successfully',
    type: DoctorCreateResponseDto,
  })
  async create(
    @Body() createDoctorDto: CreateDoctorDto,
  ): Promise<DoctorCreateResponseDto> {
    const doctor = await this.doctorService.addDoctor(createDoctorDto);
    return {
      doctorId: doctor.doctorId,
    };
  }

  /**
   * Update doctor master based on the request parameters.
   *
   * @param {UpdateDoctorRequestDto} updateDoctorRequestDto - The
   * data transfer object containing the information for updating doctor.
   * @param updateDoctorsRequestDto
   * @returns {Promise<UpdateDoctorResponseDto>} The updated
   * doctor information.
   */
  @Post('/update')
  @ApiOperation({ summary: 'Update Doctor details' })
  @ApiResponse({
    status: 200,
    description: 'Doctor Details Updated Successfully.',
    type: UpdateDoctorResponseDto,
  })
  @ApiResponse({ status: 400, description: 'Bad Request.' })
  async updateDoctors(
    @Body() updateDoctorsRequestDto: UpdateDoctorRequestDto,
  ): Promise<UpdateDoctorResponseDto> {
    return this.doctorService.updateDoctorById(updateDoctorsRequestDto);
  }

  /**
   * Fetches a paginated list of doctors based on the provided criteria.
   * @param listDoctorsRequestDto Request DTO containing
   * pagination and filter parameters
   * @returns Promise{Promise<PaginatedDoctorsResponseDto>}
   *            Returns a paginated response with:
   *          - list: Array of doctor records
   *          - totalCount: Total number of doctors
   */
  @Post('/list')
  @ApiOperation({ summary: 'Get list of doctors' })
  @ApiResponse({
    status: 200,
    description: 'Doctor list successfully fetched.',
    type: PaginatedDoctorsResponseDto,
  })
  async getDoctors(
    @Body() listDoctorsRequestDto: ListDoctorsRequestDto,
  ): Promise<PaginatedDoctorsResponseDto> {
    return this.doctorService.getDoctors(listDoctorsRequestDto);
  }
}


