/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import {
  ValidatorConstraint,
  ValidatorConstraintInterface,
  ValidationArguments,
} from 'class-validator';

/**
 * Validates that exactly one of name, hospitalId, or emailId is present.
 */
@ValidatorConstraint({ name: 'OnlyOneSearchField', async: false })
export class OnlyOneSearchField implements ValidatorConstraintInterface {
  /**
   * Custom validation to ensure that
   * exactly one of the following fields is present:
   * `hospitalName`, `hospitalId`, or `emailId`.
   *
   * This validator is useful in scenarios where only one of the search filters
   * should be allowed at a time to avoid ambiguous or conflicting queries.
   *
   * @param _ - Unused parameter, required by the method signature.
   * @param args - The validation arguments containing the target object.
   * @returns `true` if exactly one of the fields is truthy; otherwise, `false`.
   */
  validate(_: any, args: ValidationArguments): boolean {
    const obj = args.object as any;
    const fields = [obj.hospitalName, obj.hospitalId, obj.emailId];
    return fields.filter(Boolean).length === 1;
  }

  /**
   * Returns the default error message when the custom validation fails.
   *
   * This message is displayed when the input object does not have exactly one
   * of the following fields provided:
   * `hospitalName`, `hospitalId`, or `emailId`.
   *
   * @returns A string describing the validation error.
   */
  defaultMessage(): string {
    return 'Exactly one of name, hospitalId, or emailId must be provided';
  }
}

/**
 * Validates that stateId, cityId, and
 * pincodeId are allowed only if name is used.
 */
@ValidatorConstraint({ name: 'ValidLocationFieldsWithName', async: false })
export class ValidLocationFieldsWithName
implements ValidatorConstraintInterface {
  /**
   * Validates conditional presence of location-related fields.
   *
   * Ensures `stateId`, `cityId`, and `pincodeId` are only allowed when
   * `hospitalName` is provided. If `hospitalId` or `emailId` are used,
   * then location fields are not permitted.
   *
   * @param _ - Unused parameter.
   * @param args - Validation arguments providing context.
   * @returns `true` if the field combination is valid; otherwise, `false`.
   */
  validate(_: any, args: ValidationArguments): boolean {
    const obj = args.object as any;
    const { hospitalName, hospitalId, emailId, stateId, cityId, pincodeId } =
      obj;

    const locationFields = [stateId, cityId, pincodeId];
    const hasLocationData = locationFields.some((field) => !!field);

    if (hasLocationData && !hospitalName) return false;
    if ((hospitalId || emailId) && hasLocationData) return false;

    return true;
  }

  /**
 * Returns the default error message when conditional field validation fails.
 *
 * This message is used when any of the optional fields `stateId`, `cityId`,
 * or `pincodeId` are provided
 * without specifying the required `hospitalName` field.
 *
 * @returns A string indicating the conditional validation rule.
 */
  defaultMessage(): string {
    return 'stateId, cityId, pincodeId are only allowed when name is provided';
  }
}
