/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/
import { Test, TestingModule } from '@nestjs/testing';
import { HospitalService } from '../../hospital.service';
import { PrismaService } from '../../../prisma/prisma.service';
import { Decimal } from '@prisma/client/runtime/library';
import { plainToInstance, instanceToPlain } from 'class-transformer';
import { validate } from 'class-validator';
import {
  GetHospitalsRequestDto,
} from '../../dto/get-hospitals/get-hospital-list-request.dto';
import {
  GetHospitalsResponseDto } from
  '../../dto/get-hospitals/get-hospital-list-response.dto';

let service: HospitalService;
const mockPrisma = {
  hospital: {
    findMany: jest.fn(),
    count: jest.fn(),
  },
};

beforeEach(async () => {
  const module: TestingModule = await Test.createTestingModule({
    providers: [
      HospitalService,
      { provide: PrismaService, useValue: mockPrisma },
    ],
  }).compile();
  service = module.get<HospitalService>(HospitalService);
});
afterEach(() => {
  jest.clearAllMocks();
});

describe('GetHospitalsRequestDto - Transformation & Validation', () => {
  it('should transform string page and limit to numbers', async () => {
    const input = {
      page: '3',
      limit: '15',
    };

    const dtoInstance = plainToInstance(GetHospitalsRequestDto, input);
    const validationErrors = await validate(dtoInstance);

    expect(validationErrors.length).toBe(1);
    expect(dtoInstance.page).toBe(3); // string '3' -> number 3
    expect(dtoInstance.limit).toBe(15); // string '15' -> number 15
    expect(typeof dtoInstance.page).toBe('number');
    expect(typeof dtoInstance.limit).toBe('number');
  });

  it('should fail if limit is not a number', async () => {
    const input = { page: '1', limit: 'abc' };
    const dtoInstance = plainToInstance(GetHospitalsRequestDto, input);
    const validationErrors = await validate(dtoInstance);
    expect(validationErrors.some((err) => err.property === 'limit')).toBe(true);
  });
});

describe('GetHospitalsResponseDto Transformation', () => {
  it(
      'Transform plain object to responseDto with HospitalItemDto list',
      () => {
        const raw = {
          totalCount: 1,
          list: [
            {
              id: '550e8400-e29b-41d4-a716-446655440002',
              hospitalNo: 'HSP-23HD', rohiniId: 'ROH001',
              hfrId: 'HFR001', hospitalName: 'Apollo Hospitals',
              address: 'Mumbai', locality: 'Downtown',
              pincode: '667333', emailId: 'apollo@hospital.com',
              website: 'www.apollohospital.com', isHospitalRegistered: true,
              registrationNo: 'knvkj7864', registrationAuthority: 'knvkj7864',
              registeredFrom: '2025-05-30T14:45:00.000Z',
              registeredTill: '2025-12-31T23:59:59.000Z',
              latitude: 76.4878795, longitude: 76.4878795,
              contactNo: '7208806389', isEPDEnabled: true,
              isNonNetwork: true, status: 'ACTIVE',
              remarks: 'Best Hospital',
              stateId: 'dd1d8fed-9b8f-4166-96fe-7569d1f6f520',
              cityId: 'dd1d8fed-9b8f-4166-96fe-7569d1f6f520',
              pincodeId: 'dd1d8fed-9b8f-4166-96fe-7569d1f6f520',
            },
          ],
        };

        const dtoInstance = plainToInstance(GetHospitalsResponseDto, raw);
        expect(dtoInstance.totalCount).toBe(1);
        expect(dtoInstance.list).toHaveLength(1);
        expect(dtoInstance).toBeInstanceOf(GetHospitalsResponseDto);
        expect(dtoInstance.list[0].hospitalName).toBe('Apollo Hospitals');
        expect(dtoInstance.list[0].latitude).toBe(76.4878795);
        expect(dtoInstance.list[0].registeredFrom).toBe('2025-05-30 20:15:00');
      });
  it('should serialize GetHospitalsResponseDto back to plain object', () => {
    const instance = plainToInstance(GetHospitalsResponseDto, {
      totalCount: 2,
      list: [
        {
          id: '123',
          name: 'Test Hospital',
          latitude: 10.1,
          longitude: 20.2,
        },
      ],
    });

    const plain = instanceToPlain(instance);
    expect(plain.totalCount).toBe(2);
    expect(Array.isArray(plain.list)).toBe(true);
    expect(plain.list[0].name).toBe('Test Hospital');
  });
  it('should handle missing pincode_master gracefully', () => {
    const input = [
      {
        id: '123', hospital_no: '001', rohini_id: 'R001', hfr_id: 'H001',
        name: 'Test Hospital', address: '123 Street', locality: 'Area',
        pincode_id: 200, pincode_master: null,
        email_id: 'test@example.com', website: 'http://example.com',
        is_hospital_registered: true, registration_no: 'REG001',
        registration_authority: 'Health Dept', registered_from: '2020-01-01',
        registered_till: '2030-01-01', latitude: new Decimal(10.1234),
        longitude: new Decimal(20.5678), contact_hash: '123456',
        is_epd_enabled: true, is_non_network: false,
        status: 'active', remarks: 'Verified',
      },
    ];
    const result = service.transformHospitalData(input);
    expect(result[0].pincode).toBeNull();
  });
});
