/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import { Test, TestingModule } from '@nestjs/testing';
import { DoctorService } from './doctor.service';
import { PrismaService } from '../prisma/prisma.service';
jest.mock('nest-common-utilities', () => ({
  ...jest.requireActual('nest-common-utilities'),
  handlePrismaError: jest.fn((error) => {
    throw error;
  }), // default: throws error
}));

import { CreateDoctorDto } from './dto/create-doctor.dto';
import { AppException } from 'nest-common-utilities';
import { UpdateDoctorRequestDto } from './dto/update-doctor-request.dto';

let service: DoctorService;

const mockPrismaService = {
  doctor_master: {
    create: jest.fn(),
    createMany: jest.fn(),
    findUnique: jest.fn(),
    update: jest.fn(),
    findFirst: jest.fn(),
  },
  doctor_qualifications: {
    create: jest.fn(),
    createMany: jest.fn(),
  },
  pincode_master: {
    findUnique: jest.fn(),
  },
  $transaction: jest.fn(),
};


beforeEach(() => {
  mockPrismaService.doctor_master.create.mockClear();
  mockPrismaService.doctor_master.findUnique.mockClear();
  mockPrismaService.doctor_master.update.mockClear();
  mockPrismaService.doctor_qualifications.create.mockClear();
  mockPrismaService.doctor_qualifications.createMany.mockClear();
  mockPrismaService.$transaction.mockClear();

  (mockPrismaService.doctor_qualifications.create as jest.Mock).mockReturnValue(Promise.resolve({}));
  (mockPrismaService.doctor_master.create as jest.Mock).mockReturnValue(Promise.resolve({ id: '48c48258-5ed4-48be-a1f9-44be7bbeecdd' }));
  (mockPrismaService.$transaction as jest.Mock).mockImplementation(async (cb) => {
    return cb(mockPrismaService);
  });
});


beforeEach(async () => {
  const module: TestingModule = await Test.createTestingModule({
    providers: [
      DoctorService,
      {
        provide: PrismaService,
        useValue: mockPrismaService,
      },
    ],
  }).compile();
  service = module.get<DoctorService>(DoctorService);
});

describe('DoctorService', () => {
  it('should be defined', () => {
    expect(service).toBeDefined();
  });
  describe('updateDoctor', () => {
    const dto: UpdateDoctorRequestDto = {
      doctorId: 'uuid-old-doc',
      userId: 'uuid-user',
      doctorName: 'Dr. Updated',
      doctorRegNo: 'REG111',
      doctorSpecialityId: 'uuid-spec',
      doctorQualificationIds: ['qual-1', 'qual-2'],
      contactNo: '9999999999',
      hprCode: 'HPR-111',
      emailId: 'updated@abc.com',
      panNo: 'ABCDE9876F',
      address: 'Updated Addr',
      pincodeId: 'uuid-pin',
      latitude: 12.111,
      longitude: 77.222,
    };

    it('should throw AppException if doctor with same regNo or hprCode exists', async () => {
      (mockPrismaService.doctor_master.findFirst as jest.Mock).mockResolvedValue({ id: 'uuid-existing' });

      await expect(service.updateDoctorById(dto)).rejects.toThrow(AppException);
      expect(mockPrismaService.doctor_master.findFirst).toHaveBeenCalledWith({
        where: {
          OR: [{ registration_no: dto.doctorRegNo }, { hpr_code: dto.hprCode }],
        },
      });
    });

    it('should call handlePrismaError if transaction fails', async () => {
      const error = new Error('Some Prisma Error');

      (
        mockPrismaService.doctor_master.findFirst as jest.Mock
      ).mockResolvedValue(null);
      (
        mockPrismaService.pincode_master.findUnique as jest.Mock
      ).mockResolvedValue({
        state_id: 'state-id',
        city_id: 'city-id',
      });

      (mockPrismaService.doctor_master.update as jest.Mock).mockRejectedValue(
          error,
      );

      const handlePrismaSpy = jest.spyOn(require('nest-common-utilities'), 'handlePrismaError').mockImplementation(() => {});

      await service.updateDoctorById(dto);

      expect(handlePrismaSpy).toHaveBeenCalledWith(error);
    });

    it('should deactivate old doctor and create new one with qualifications', async () => {
      (mockPrismaService.doctor_master.findFirst as jest.Mock).mockResolvedValue(null);
      (mockPrismaService.pincode_master.findUnique as jest.Mock).mockResolvedValue({
        state_id: 'state-id',
        city_id: 'city-id',
      });
      (mockPrismaService.doctor_master.update as jest.Mock).mockResolvedValue({ id: dto.doctorId });
      const newDoctorRecord = { id: 'uuid-new-doc' };
      const tx = {
        doctor_master: {
          create: jest.fn().mockResolvedValue(newDoctorRecord),
        },
        doctor_qualifications: {
          createMany: jest.fn(),
        },
      };
      (mockPrismaService.$transaction as jest.Mock).mockImplementation((cb) => cb(tx));

      const result = await service.updateDoctorById(dto);

      expect(mockPrismaService.doctor_master.update).toHaveBeenCalledWith({
        where: { id: dto.doctorId },
        data: {
          is_active: false,
          updated_by: dto.userId,
          updated_at: expect.any(Date),
        },
      });

      expect(tx.doctor_master.create).toHaveBeenCalledWith({
        data: expect.objectContaining({
          name: dto.doctorName,
          registration_no: dto.doctorRegNo,
          speciality_id: dto.doctorSpecialityId,
          doctor_contact: dto.contactNo,
          hpr_code: dto.hprCode,
          doctor_email_id: dto.emailId,
          pan_no: dto.panNo,
          address: dto.address,
          pincode_id: dto.pincodeId,
          latitude: dto.latitude,
          longitude: dto.longitude,
          state_id: 'state-id',
          city_id: 'city-id',
          created_by: dto.userId,
          updated_by: dto.userId,
          // is_active: true,
        }),
      });

      expect(tx.doctor_qualifications.createMany).toHaveBeenCalledWith({
        data: [
          {
            doctor_id: 'uuid-new-doc',
            doctor_qualification_id: 'qual-1',
            created_by: dto.userId,
            updated_by: dto.userId,
          },
          {
            doctor_id: 'uuid-new-doc',
            doctor_qualification_id: 'qual-2',
            created_by: dto.userId,
            updated_by: dto.userId,
          },
        ],
      });

      expect(result).toEqual({ doctorId: 'uuid-new-doc' });
    });

    it('should create doctor with no qualifications if doctorQualificationIds is empty', async () => {
      const dtoWithoutQualifications = {
        ...dto,
        doctorQualificationIds: [],
      };

      (mockPrismaService.doctor_master.findFirst as jest.Mock).mockResolvedValue(null);
      (mockPrismaService.pincode_master.findUnique as jest.Mock).mockResolvedValue({
        state_id: null,
        city_id: null,
      });
      (mockPrismaService.doctor_master.update as jest.Mock).mockResolvedValue({ id: dto.doctorId });

      const newDoctorRecord = { id: 'uuid-new-doc' };
      const tx = {
        doctor_master: {
          create: jest.fn().mockResolvedValue(newDoctorRecord),
        },
        doctor_qualifications: {
          createMany: jest.fn(),
        },
      };
      (mockPrismaService.$transaction as jest.Mock).mockImplementation((cb) => cb(tx));

      const result = await service.updateDoctorById(dtoWithoutQualifications);

      expect(tx.doctor_qualifications.createMany).not.toHaveBeenCalled();
      expect(result).toEqual({ doctorId: 'uuid-new-doc' });
    });
  });
});

describe('addDoctor', () => {
  it('should create a doctor with all fields and qualifications', async () => {
    const dto: any = {
      doctorName: 'Dr. Full Fields',
      doctorRegNo: 'REG123',
      doctorSpecialityId: 'spec-uuid',
      contactNo: '1234567890',
      hprCode: 'HPR001',
      emailId: 'full@example.com',
      panNo: 'ABCDE1234F',
      address: '123 Test Street',
      pincodeId: 'pin-uuid',
      latitude: 12.34,
      longitude: 56.78,
      created_by: 'creator-uuid',
      updated_by: 'updater-uuid',
      doctorQualificationIds: ['qual-1', 'qual-2'],
    };
    (mockPrismaService.doctor_master.create as jest.Mock).mockResolvedValue({
      id: 'doc-id',
    });
    (
      mockPrismaService.doctor_qualifications.createMany as jest.Mock
    ).mockResolvedValue({ count: 2 });
    (mockPrismaService.doctor_master.findFirst as jest.Mock).mockResolvedValue(null);
    (mockPrismaService.pincode_master.findUnique as jest.Mock).mockResolvedValue({
      state_id: 'state-id',
      city_id: 'city-id',
    });
    const result = await service.addDoctor(dto);

    expect(mockPrismaService.doctor_master.create).toHaveBeenCalledWith(expect.objectContaining({
      data: expect.objectContaining({
        registration_no: dto.doctorRegNo,
        doctor_contact: dto.contactNo,
        hpr_code: dto.hprCode,
        doctor_email_id: dto.emailId,
        pan_no: dto.panNo,
        address: dto.address,
        pincode_id: dto.pincodeId,
        latitude: dto.latitude,
        longitude: dto.longitude,
        created_by: dto.created_by,
        updated_by: dto.updated_by,
        name: dto.doctorName,
        speciality_id: dto.doctorSpecialityId,
      }),
      select: { id: true },
    }));

    expect(
        mockPrismaService.doctor_qualifications.createMany,
    ).toHaveBeenCalledWith({
      data: [
        {
          doctor_id: 'doc-id',
          doctor_qualification_id: 'qual-1',
          created_by: 'creator-uuid',
          updated_by: 'updater-uuid',
        },
        {
          doctor_id: 'doc-id',
          doctor_qualification_id: 'qual-2',
          created_by: 'creator-uuid',
          updated_by: 'updater-uuid',
        },
      ],
    });

    expect(result).toEqual({ doctorId: 'doc-id' });
  });

  it(
      `should create a doctor with only required fields, 
    covering ?? null and updated_by fallback`,
      async () => {
        const dto: any = {
          doctorName: 'Dr. Minimal',
          doctorSpecialityId: 'spec-uuid',
          created_by: 'creator-uuid',
          // doctorQualificationIds: undefined, // explicitly undefined or omitted
        };
        (mockPrismaService.doctor_master.create as jest.Mock).mockResolvedValue({
          id: 'doc-id',
        });
        (
      mockPrismaService.doctor_qualifications.createMany as jest.Mock
        ).mockResolvedValue({ count: 0 });

        const result = await service.addDoctor(dto);

        expect(mockPrismaService.doctor_master.create).toHaveBeenCalledWith(
            expect.objectContaining({
              data: expect.objectContaining({
                registration_no: null,
                doctor_contact: null,
                hpr_code: null,
                doctor_email_id: null,
                pan_no: null,
                address: null,
                pincode_id: null,
                latitude: null,
                longitude: null,
                created_by: dto.created_by,
                updated_by: dto.created_by,
                name: dto.doctorName,
                speciality_id: dto.doctorSpecialityId,
              }),
              select: { id: true },
            }),
        );

        expect(
            mockPrismaService.doctor_qualifications.createMany,
        ).toHaveBeenCalledTimes(0);
        expect(result).toEqual({ doctorId: 'doc-id' });
      });

  const baseDoctorDto: CreateDoctorDto = {
    doctorName: 'Dr. Test User', emailId: 'test.user@example.com',
    contactNo: '1234567890', doctorRegNo: 'REG123',
    hprCode: 'HPR001', panNo: 'ABCDE1234F',
    address: '123 Test Street', pincodeId: 'uuid-pincode',
    doctorQualificationIds: [], doctorSpecialityId: 'uuid-spec',
    latitude: 12.9716, longitude: 77.5946,
    created_by: 'creator-uuid', updated_by: 'updater-uuid',
  };

  it('should create a doctor and return the doctorId', async () => {
    const result = await service.addDoctor(baseDoctorDto);
    expect(mockPrismaService.doctor_master.create).toHaveBeenCalledWith({
      data: expect.objectContaining({
        name: baseDoctorDto.doctorName,
        registration_no: baseDoctorDto.doctorRegNo,
        speciality_id: baseDoctorDto.doctorSpecialityId,
        doctor_contact: baseDoctorDto.contactNo,
        hpr_code: baseDoctorDto.hprCode,
        doctor_email_id: baseDoctorDto.emailId,
        pan_no: baseDoctorDto.panNo,
        address: baseDoctorDto.address,
        pincode_id: baseDoctorDto.pincodeId,
        latitude: baseDoctorDto.latitude,
        longitude: baseDoctorDto.longitude,
        created_by: baseDoctorDto.created_by,
        updated_by: baseDoctorDto.updated_by,
      }),
      select: { id: true },
    });
    expect(result).toEqual({ doctorId: '48c48258-5ed4-48be-a1f9-44be7bbeecdd' });
  });
  it('should throw an error if doctor creation fails', async () => {
    const { handlePrismaError } = require('nest-common-utilities');
    (handlePrismaError as jest.Mock).mockImplementation(() => {
      throw new Error('MOCK_HANDLE_PRISMA_ERROR');
    });
    (mockPrismaService.$transaction as jest.Mock).mockRejectedValue(
        new Error('Prisma error'),
    );
    const minimalDto: CreateDoctorDto = {
      doctorName: 'Dr. Fail', created_by: 'creator-fail',
      updated_by: 'updater-fail',
      doctorSpecialityId: 'uuid-spec',
    };
    await expect(service.addDoctor(minimalDto)).rejects.toThrow('MOCK_HANDLE_PRISMA_ERROR');
  });
});
