/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import { NestFactory, Reflector } from '@nestjs/core';
import {
  ClassSerializerInterceptor,
  ValidationPipe,
  VersioningType,
} from '@nestjs/common';
import { AppModule } from './app.module';
import helmet from 'helmet';
import { setupSwagger } from './config/swagger.config';
import {
  Logger,
  GlobalExceptionFilter,
  ResponseInterceptor,
  setPiiFields,
  AppException,
  fetchAllSecretsAndWriteEnv,
} from 'nest-common-utilities';
import { MicroserviceOptions, Transport } from '@nestjs/microservices';
import { config } from './config/pii.config';
import * as dotenv from 'dotenv';

/**
 * The main function that bootstraps the NestJS application,
 * sets up global middleware,
 * exception filters, interceptors,
 * CORS, security headers, and Swagger documentation.
 *
 * @returns {Promise<void>} A promise that resolves
 * once the application has started.
 */
async function bootstrap(): Promise<void> {
  const logger = new Logger('app');
  await fetchAllSecretsAndWriteEnv().catch((err) => {
    logger.error('error fetching secrets: ' + err);
  });

  dotenv.config();

  const app = await NestFactory.create(AppModule);

  setPiiFields(config);

  // Global filters and interceptors
  app.useGlobalFilters(new GlobalExceptionFilter());
  app.useGlobalInterceptors(new ClassSerializerInterceptor(app.get(Reflector)));
  app.useGlobalInterceptors(new ResponseInterceptor());

  app.useGlobalPipes(
      new ValidationPipe({
        whitelist: true, // strip unknown fields
        forbidNonWhitelisted: true, // throw error on unknown fields
        transform: true, // auto-transform payloads to DTOs
        exceptionFactory: (errors) => {
          const messages = errors.flatMap((err) =>
            Object.values(err.constraints || {}).map((msg) => ({
              key: err.property,
              message: msg,
            })),
          );
          const combinedMessage = messages.map((m) =>
            `${m.key}: ${m.message}`,
          ).join(', ');
          return new AppException(
              combinedMessage || 'Validation failed',
              'VALIDATION_ERROR',
          );
        },
      }),
  );

  // Security and CORS
  app.use(helmet());
  app.enableCors();

  // Swagger documentation (enabled in non-prod environments)
  if (process.env.NODE_ENV !== 'prod') {
    setupSwagger(app);
  }

  // Microservice setup using TCP transport
  const microservice =
    await NestFactory.createMicroservice<MicroserviceOptions>(AppModule, {
      transport: Transport.TCP,
      options: {
        host: '0.0.0.0',
        port: 8005,
      },
    });

  // Global filters and interceptors for microservice
  microservice.useGlobalFilters(new GlobalExceptionFilter());
  microservice.useGlobalInterceptors(
      new ClassSerializerInterceptor(microservice.get(Reflector)),
  );
  microservice.useGlobalInterceptors(new ResponseInterceptor());
  microservice.useGlobalPipes(
      new ValidationPipe({
        whitelist: true, // strip unknown fields
        forbidNonWhitelisted: true, // throw error on unknown fields
        transform: true, // auto-transform payloads to DTOs
        exceptionFactory: (errors) => {
          const messages = errors.flatMap((err) =>
            Object.values(err.constraints || {}).map((msg) => ({
              key: err.property,
              message: msg,
            })),
          );
          const combinedMessage = messages.map((m) =>
            `${m.key}: ${m.message}`,
          ).join(', ');
          return new AppException(
              combinedMessage || 'Validation failed',
              'VALIDATION_ERROR',
          );
        },
      }),
  );

  await microservice.listen();
  logger.info(`Microservice started on port: 8005`);

  // Global versioning
  app.setGlobalPrefix('hospital/api');
  app.enableVersioning({
    type: VersioningType.URI,
    defaultVersion: '1',
  });

  // Start HTTP server
  const port = process.env.PORT || 3005;
  await app.listen(port, () => {
    logger.info(`Server started on port: ${port}`);
  });
}

// Start the app
bootstrap();
