/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import { Injectable } from '@nestjs/common';
import { CreateDoctorDto } from './dto/create-doctor.dto';
import { PrismaService } from '../prisma/prisma.service';
import {
  AppException,
  errorCode,
  handlePrismaError,
  Logger,
  fuzzySearch
} from 'nest-common-utilities';
import { UpdateDoctorRequestDto } from './dto/update-doctor-request.dto';
import { UpdateDoctorResponseDto } from './dto/update-doctor-response.dto';
import { ListDoctorsRequestDto } from './dto/list-doctors-request.dto';
import {
  PaginatedDoctorsResponseDto,
} from './dto/paginated-doctors-response.dto';

/**
 * Service to encapsulate doctors
 * domain logic such as create doctor update doctor.
 */
@Injectable()
export class DoctorService {
  private logger = new Logger(DoctorService.name);

  /**
   * Constructs the DoctorService.
   *
   * @param prisma - The Prisma service for database interactions.
   */
  constructor(private readonly prisma: PrismaService) { }

  /**
   * Calculates pagination parameters based on the provided page and limit.
   *
   * @param page - Page number (defaults to 1 if not provided or invalid).
   * @param limit - Number of records per page (defaults to 10 if invalid).
   * @returns An object containing `limit` and `offset` values for pagination.
   */
  async pagination(
    page?: number,
    limit?: number,
  ): Promise<{ limit: number; offset: number }> {
    const safeLimit = limit && limit > 0 ? limit : 10;
    const safePage = page && page > 0 ? page : 1;
    const offset = (safePage - 1) * safeLimit;
    return { limit: safeLimit, offset };
  }

  /**
   * Create the doctor details by using doctor ID
   *
   *
   *
   * @param {CreateDoctorDto} payload - The data transfer
   * object containing the doctor information to be updated.
   * @returns {Promise<DoctorCreateResponseDto>} A promise
   * that resolves to the created doctor's data.
   */
  /**
   * Add a new doctor and their qualifications.
   * @param payload - CreateDoctorDto
   * @returns Standard response with new doctor id
   */
  async addDoctor(payload: CreateDoctorDto) {
    try {
      if (payload.doctorRegNo || payload.hprCode) {
        const existingDoctor = await this.prisma.doctor_master.findFirst({
          where: {
            OR: [
              { registration_no: payload.doctorRegNo },
              { hpr_code: payload.hprCode },
            ],
          },
        });
        if (existingDoctor) {
          throw new AppException(
            'A doctor with the same Registration No or HPR Code already exists',
            400,
          );
        }
      }

      // Resolve city_id and state_id from pincode_master
      let resolvedCityId: string | null = null;
      let resolvedStateId: string | null = null;

      if (payload.pincodeId) {
        const pincodeRecord = await this.prisma.pincode_master.findUnique({
          where: { id: payload.pincodeId },
          select: { city_id: true, state_id: true },
        });
        if (pincodeRecord) {
          resolvedCityId = pincodeRecord.city_id;
          resolvedStateId = pincodeRecord.state_id;
        }
      }
      return await this.prisma.$transaction(async (tx) => {
        const newDoctor = await tx.doctor_master.create({
          data: {
            name: payload.doctorName,
            registration_no: payload.doctorRegNo ?? null,
            speciality_id: payload.doctorSpecialityId,
            doctor_contact: payload.contactNo ?? null,
            hpr_code: payload.hprCode ?? null,
            doctor_email_id: payload.emailId ?? null,
            // @ts-ignore
            pan_no: payload.panNo ?? null,
            address: payload.address ?? null,
            pincode_id: payload.pincodeId ?? null,
            latitude: payload.latitude ?? null,
            longitude: payload.longitude ?? null,
            created_by: payload.created_by,
            updated_by: payload.updated_by ?? payload.created_by,
            city_id: resolvedCityId,
            state_id: resolvedStateId,
          },
          select: { id: true },
        });

        const qualificationIds = payload.doctorQualificationIds ?? [];
        if (Array.isArray(qualificationIds) && qualificationIds.length > 0) {
          const qualificationData = qualificationIds.map((id) => ({
            doctor_id: newDoctor.id,
            doctor_qualification_id: id,
            created_by: payload.created_by,
            updated_by: payload.updated_by ?? payload.created_by,
          }));

          await tx.doctor_qualifications.createMany({
            data: qualificationData,
          });
        }

        return { doctorId: newDoctor.id };
      });
    } catch (error) {
      handlePrismaError(error);
    }
  }

  /**
   * Updates an existing doctor record by deactivating the old entry and
   * creating a new one with updated details.
   *
   * This method checks if the doctor with the given ID exists. If not,
   * it throws an exception. If the doctor exists, it marks the current
   * record as inactive, then inserts a new record with the updated data.
   *
   * @param {UpdateDoctorRequestDto} dto - The DTO containing updated
   * doctor information.
   *
   * @returns {Promise<UpdateDoctorResponseDto>} The response containing
   * the ID of the newly created doctor record.
   *
   * @throws {AppException} If no doctor is found with the given ID.
   */
  async updateDoctorById(
    dto: UpdateDoctorRequestDto,
  ): Promise<UpdateDoctorResponseDto> {
    const {
      doctorId,
      userId,
      doctorName,
      doctorRegNo,
      doctorSpecialityId,
      doctorQualificationIds,
      contactNo,
      hprCode,
      emailId,
      panNo,
      address,
      pincodeId,
      latitude,
      longitude,
    } = dto;
    try {
      if (doctorRegNo || hprCode) {
        // Check for duplicate doctorRegNo or hprCode within hospital
        const existingDoctor = await this.prisma.doctor_master.findFirst({
          where: {
            OR: [{ registration_no: dto.doctorRegNo }, { hpr_code: dto.hprCode }],
          },
        });
        if (existingDoctor?.id) {
          throw new AppException(
            `Doctor with ID ${doctorId} is already exists.`,
            errorCode.DATA_NOT_FOUND,
          );
        }
      }

      // Resolve city_id and state_id from pincode_master
      let resolvedCityId: string | null = null;
      let resolvedStateId: string | null = null;

      if (pincodeId) {
        const pincodeRecord = await this.prisma.pincode_master.findUnique({
          where: { id: pincodeId },
          select: { city_id: true, state_id: true },
        });
        if (pincodeRecord) {
          resolvedCityId = pincodeRecord.city_id;
          resolvedStateId = pincodeRecord.state_id;
        }
      }

      const data = {
        name: doctorName,
        registration_no: doctorRegNo ?? null,
        speciality_id: doctorSpecialityId,
        doctor_contact: contactNo ?? null,
        hpr_code: hprCode ?? null,
        doctor_email_id: emailId ?? null,
        //@ts-ignore
        pan_no: panNo ?? null,
        address: address ?? null,
        pincode_id: pincodeId ?? null,
        latitude: latitude ?? null,
        longitude: longitude ?? null,
        state_id: resolvedStateId,
        city_id: resolvedCityId,
        created_by: userId ?? '',
        updated_by: userId ?? '',
      };
      return await this.prisma.$transaction(async (tx) => {
        // Deactivate the old doctor record
        await this.prisma.doctor_master.update({
          where: { id: doctorId },
          data: {
            is_active: false,
            updated_by: userId,
            updated_at: new Date(),
          },
        });
        // @ts-ignore
        const newDoctor = await tx.doctor_master.create({ data });
        const qualificationIds = doctorQualificationIds ?? [];
        if (Array.isArray(qualificationIds) && qualificationIds.length > 0) {
          const qualificationData = qualificationIds.map((id) => ({
            doctor_id: newDoctor.id,
            doctor_qualification_id: id,
            created_by: userId ?? '',
            updated_by: userId ?? '',
          }));

          await tx.doctor_qualifications.createMany({
            data: qualificationData,
          });
        }
        return {
          doctorId: newDoctor.id,
        };
      });
    } catch (error) {
      handlePrismaError(error);
    }
  }

  /**
   * Fetches a list of doctors based on the provided filters.
   * @param listDoctorsRequestDto - The DTO containing filter parameters.
   * @returns {Promise<PaginatedDoctorsResponseDto>}
   * A promise that resolves to the list of doctors.
   */
  async getDoctors(
    listDoctorsRequestDto: ListDoctorsRequestDto,
  ): Promise<PaginatedDoctorsResponseDto> {
    try {
      const { doctorName, doctorRegNo, specialityId, emailId, mobileNo, hprCode, panNo, page, limit } =
        listDoctorsRequestDto;

      const whereCondition: Record<string, any> = {};

      if (specialityId) whereCondition.speciality_id = specialityId;
      if (doctorRegNo) whereCondition.registration_no = doctorRegNo;
      if (emailId) whereCondition.doctor_email_id = emailId;
      if (mobileNo) whereCondition.doctor_contact = mobileNo;
      if (hprCode) whereCondition.hpr_code = hprCode;
      if (panNo) whereCondition.pan_no = panNo;

      const { offset: skip, limit: take } = await this.pagination(page, limit);

      const data = await fuzzySearch<any>(this.prisma, 'doctor_master', {
        searchFields: ['name'],
        searchTerm: doctorName || '',
        limit: take,
        offset: skip,
        similarityThreshold: 0.1,
        where: whereCondition,
        include: {
          doctor_speciality_master: true,
          qualifications: {
            select: {
              id: true,
              qualification: {
                select: { id: true, name: true },
              },
            },
          },
          pincode_master: true,
        },
      });

      const transformedDoctors = data.data.map((doctor) => ({
        id: doctor.id ?? '',
        doctorName: doctor.name ?? '',
        doctorRegNo: doctor.registration_no ?? '',
        specialityId: doctor.speciality_id ?? '',
        speciality: doctor.doctor_speciality_master?.name ?? '',
        qualificationIds:
          doctor.qualifications?.map((qualification) => ({
            id: qualification.id,
            qualification: qualification.qualification.name,
          })) ?? [],
        contactNo: doctor.doctor_contact ?? '',
        hprCode: doctor.hpr_code ?? '',
        emailId: doctor.doctor_email_id ?? null,
        panNo: doctor.pan_no ? doctor.pan_no.toString() : '',
        address: doctor.address ?? '',
        pincodeId: doctor.pincode_master?.id ?? '',
        pincode: doctor.pincode_master?.name ?? '',
        latitude: doctor.latitude?.toString() ?? null,
        longitude: doctor.longitude?.toString() ?? null,
      }));
      return {
        totalCount: data.total,
        data: transformedDoctors,
      };
    }
    catch (error) {
      console.log("error", error);
      this.logger.error('Error fetching doctor:', error);
      handlePrismaError(error);
    }
  }

}
