/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import { Test, TestingModule } from '@nestjs/testing';
import { DoctorController } from './doctor.controller';
import { DoctorService } from './doctor.service';
import { CreateDoctorDto } from './dto/create-doctor.dto';
import { HttpException, HttpStatus } from '@nestjs/common';
import { DoctorCreateResponseDto } from './dto/doctor-dto';
import { UpdateDoctorRequestDto } from './dto/update-doctor-request.dto';
import { UpdateDoctorResponseDto } from './dto/update-doctor-response.dto';

let controller: DoctorController;

const mockDoctorService = {
  addDoctor: jest.fn(),
  updateDoctorById: jest.fn(),
};

beforeEach(async () => {
  const module: TestingModule = await Test.createTestingModule({
    controllers: [DoctorController],
    providers: [
      {
        provide: DoctorService,
        useValue: mockDoctorService,
      },
    ],
  }).compile();

  controller = module.get<DoctorController>(DoctorController);
});

describe('DoctorController', () => {
  it('should be defined', () => {
    expect(controller).toBeDefined();
  });
  describe('create', () => {
    it('should create a doctor and return success with doctorId', async () => {
      const createDoctorDto: CreateDoctorDto = {
        doctorName: 'Dr. Test',
        doctorQualificationIds: ['0ec006f7-5da5-4f03-843c-fcddbd8e47ba', '830e2305-7b54-4766-8145-a89e67d597f4'],
        doctorSpecialityId: 'spec-uuid-abc',
        emailId: 'dr.test@example.com',
        contactNo: '1234567890', doctorRegNo: 'REG12345',
        hprCode: 'HPR123', panNo: 'ABCDE1234F',
        address: '123 Test St', pincodeId: 'pincode-uuid-123',
        latitude: 12.345678, longitude: 78.901234,
        created_by: 'user-creator-uuid',
        updated_by: 'user-updater-uuid',
      };
      const expectedDoctorId = 'doc-uuid-123';
      mockDoctorService.addDoctor.mockResolvedValue({
        doctorId: expectedDoctorId,
      });
      const result: DoctorCreateResponseDto =
        await controller.create(createDoctorDto);
      expect(mockDoctorService.addDoctor).toHaveBeenCalledWith(createDoctorDto);
      expect(result).toEqual({
        doctorId: expectedDoctorId,
      });
    });
    it('should return error response if service throws an error', async () => {
      const createDoctorDto: CreateDoctorDto = {
        doctorName: 'Dr. Error',
        doctorQualificationIds: ['0ec006f7-5da5-4f03-843c-fcddbd8e47ba', '830e2305-7b54-4766-8145-a89e67d597f4'],
        doctorSpecialityId: '0ec006f7-5da5-4f03-843c-fcddbd8e47ba',
        emailId: 'dr.error@example.com',
        contactNo: '0987654321', doctorRegNo: 'REG67890',
        hprCode: 'HPR456', panNo: 'FGHIJ5678K',
        address: '456 Error Ave', pincodeId: 'pincode-uuid-456',
        latitude: 21.345678, longitude: 87.901234,
        created_by: 'user-creator-err-uuid',
        updated_by: 'user-updater-err-uuid',
      };
      const errorResponse = new HttpException(
          'Could not create doctor.',
          HttpStatus.INTERNAL_SERVER_ERROR,
      );
      mockDoctorService.addDoctor.mockRejectedValue(errorResponse);
      await expect(controller.create(createDoctorDto)).rejects.toThrow(
          errorResponse,
      );
    });
    it(
        'should throw Exception with CONFLICT status if reg number exists',
        async () => {
          const createDoctorDto: CreateDoctorDto = {
            doctorName: 'Dr. Duplicate',
            doctorQualificationIds: ['0ec006f7-5da5-4f03-843c-fcddbd8e47ba', '830e2305-7b54-4766-8145-a89e67d597f4'],
            doctorSpecialityId: '0ec006f7-5da5-4f03-843c-fcddbd8e47ba',
            emailId: 'dr.duplicate@example.com',
            contactNo: '1122334455', doctorRegNo: 'REG_DUPLICATE',
            hprCode: 'HPR_DUP', panNo: 'PQRST1234Z',
            address: '789 Duplicate Ln',
            pincodeId: 'pincode-uuid-789',
            latitude: 34.56789, longitude: 98.765432,
            created_by: 'user-creator-dup-uuid',
            updated_by: 'user-updater-dup-uuid',
          };
          const errorResponse = new HttpException(
              'Registration number already exists.',
              HttpStatus.CONFLICT,
          );
          mockDoctorService.addDoctor.mockRejectedValue(errorResponse);
          await expect(controller.create(createDoctorDto)).rejects.toThrow(
              errorResponse,
          );
        });
    it(
        'should throw Exception with CONFLICT status for unique violations',
        async () => {
          const createDoctorDto: CreateDoctorDto = {
            doctorName: 'Dr. UniqueFail',
            doctorQualificationIds: ['0ec006f7-5da5-4f03-843c-fcddbd8e47ba', '830e2305-7b54-4766-8145-a89e67d597f4'],
            doctorSpecialityId: '0ec006f7-5da5-4f03-843c-fcddbd8e47ba',
            emailId: 'dr.uniquefail@example.com',
            contactNo: '2233445566',
            doctorRegNo: 'REG_UNIQUE_FAIL',
            hprCode: 'HPR_UF', panNo: 'LMNOP6789Y',
            address: '101 Unique St',
            pincodeId: 'pincode-uuid-101',
            latitude: 45.678901, longitude: 10.123456,
            created_by: 'user-creator-uf-uuid',
            updated_by: 'user-updater-uf-uuid',
          };
          const errorResponse = new HttpException(
              'A record with this value already exists for a unique field.',
              HttpStatus.CONFLICT,
          );
          mockDoctorService.addDoctor.mockRejectedValue(errorResponse);

          await expect(controller.create(createDoctorDto)).rejects.toThrow(
              errorResponse,
          );
        });
  });
});

describe('updateDoctors', () => {
  it('should update doctor and return the updated doctorId', async () => {
    const dto: UpdateDoctorRequestDto = {
      doctorId: '53f12c09-7be7-407b-8c5d-26892a354c26',
      userId: '6f6b90a9-331e-4874-a374-ebf65a1bce2a',
      doctorName: 'Dr. Prathviraj Thokal',
      doctorRegNo: 'RED-1111',
      doctorSpecialityId: 'b9fc627e-93f3-4b93-9ac3-3f3c3ad1e007',
      doctorQualificationIds: ['58c1b3f1-0007-4444-aaaa-000000000007'],
      contactNo: '7208806389',
      hprCode: 'HPR-111', emailId: 'prathviraj@abc.com',
      panNo: 'SLEUS7686D', address: 'Mumbai',
      pincodeId: '2c1fe645-989c-4705-91ef-36f3c78c0d08',
      latitude: 73.383683, longitude: 82.2882872,
    };
    const response: UpdateDoctorResponseDto = {
      doctorId: 'new-doctor-uuid',
    };
    (mockDoctorService.updateDoctorById as jest.Mock).mockResolvedValue(
        response,
    );
    const result = await controller.updateDoctors(dto);

    expect(mockDoctorService.updateDoctorById).toHaveBeenCalledWith(dto);
    expect(result).toEqual(response);
  });
});
