import * as fs from 'fs';
import * as path from 'path';

const inputPath = 'prisma/introspected.prisma';
const outputDir = 'prisma/models';

if (!fs.existsSync(outputDir)) {
  fs.mkdirSync(outputDir);
}

const schema = fs.readFileSync(inputPath, 'utf-8');

// Extract generator/datasource blocks (keep in main file)
const headerBlocks = schema.match(/(?:generator|datasource)[\s\S]+?\{[\s\S]+?\}/g)?.join('\n\n') ?? '';

// Extract all enums
const enumRegex = /enum\s+(\w+)\s+\{[^}]+\}/g;
const enums: Record<string, string> = {};
let enumMatch: RegExpExecArray | null;

while ((enumMatch = enumRegex.exec(schema)) !== null) {
  enums[enumMatch[1]] = enumMatch[0];
}

// Extract all composite types (Prisma calls them "types", used like value objects)
const typeRegex = /type\s+(\w+)\s+\{[^}]+\}/g;
const types: Record<string, string> = {};
let typeMatch: RegExpExecArray | null;

while ((typeMatch = typeRegex.exec(schema)) !== null) {
  types[typeMatch[1]] = typeMatch[0];
}

// Extract all models
const modelRegex = /model\s+(\w+)\s+\{[^}]+\}/g;
let modelMatch: RegExpExecArray | null;
const processedModels: string[] = [];

while ((modelMatch = modelRegex.exec(schema)) !== null) {
  const modelName = modelMatch[1];
  const modelBlock = modelMatch[0];
  processedModels.push(modelName);

  const usedItems = new Set<string>();

  // Collect dependencies (types & enums)
  for (const name of [...Object.keys(enums), ...Object.keys(types)]) {
    const usageRegex = new RegExp(`\\b${name}\\b`);
    if (usageRegex.test(modelBlock)) {
      usedItems.add(name);
    }
  }

  const dependencyBlocks = [...usedItems].map(name =>
    enums[name] ?? types[name] ?? ''
  );

  let enhancedModelBlock = modelBlock;
  enhancedModelBlock = enhancedModelBlock.replace(
    /^(\s*)(\w+)\s+String(.*?)$/gm,
    (match, indent, field, rest) => {
      const isIdField = field.toLowerCase() === 'id' || field.toLowerCase().endsWith('_id');
      const alreadyHasId = /@id/.test(rest);
      const alreadyHasDefault = /@default/.test(rest);
      const alreadyHasDb = /@db\./.test(rest);

      if (isIdField && alreadyHasId) {
        return `${indent}${field} String @id @default(uuid()) @db.Uuid`;
      }
      return match;
    }
  );

  // Enhance createdAt field
  enhancedModelBlock = enhancedModelBlock.replace(
    /^(\s*)(created_at|createdAt)\s+DateTime(.*?)$/gim,
    (match, indent, field, rest) => {
      const alreadyHasDefault = /@default\(.+\)/.test(rest);
      if (!alreadyHasDefault) {
        return `${indent}${field}  DateTime @default(now())`;
      }
      return match;
    }
  );

  // Enhance updatedAt field
  enhancedModelBlock = enhancedModelBlock.replace(
    /^(\s*)(updated_at|updatedAt)\s+DateTime(.*?)$/gim,
    (match, indent, field, rest) => {
      const alreadyHasUpdatedAt = /@updatedAt/.test(rest);
      if (!alreadyHasUpdatedAt) {
        return `${indent}${field} DateTime @updatedAt`;
      }
      return match;
    }
  );

  enhancedModelBlock = alignModelBlock(enhancedModelBlock);
  const fileContent = [...dependencyBlocks, enhancedModelBlock].join('\n\n');
  const outputPath = path.join(outputDir, `${modelName}.prisma`);

  fs.writeFileSync(outputPath, fileContent, 'utf-8');
  console.log(`Created/Updated: models/${modelName}.prisma`);
}

// Optional: Clean up stale model files
// const existingFiles = fs.readdirSync(outputDir);
// for (const file of existingFiles) {
//   const name = path.parse(file).name;
//   if (!processedModels.includes(name)) {
//     fs.unlinkSync(path.join(outputDir, file));
//     console.log(`Removed unused model file: ${file}`);
//   }
// }

// Clear introspected.prisma (preserve generator/datasource)
// fs.writeFileSync(inputPath, headerBlocks + '\n');
fs.writeFileSync(inputPath, '' + '\n');
console.log(`Cleaned introspected.prisma (preserved generator & datasource)`);


function alignModelBlock(block: string): string {
  const lines = block.trim().split('\n');

  const modelStart = lines[0]; // should be `model Something {`
  const modelEnd = lines[lines.length - 1]; // should be `}`

  const fieldLines = lines.slice(1, -1);

  let maxField = 0;
  let maxType = 0;

  for (const line of fieldLines) {
    const match = line.trim().match(/^(\w+)\s+([^\s\[\]]+)(\s|\[|$)/);
    if (match) {
      maxField = Math.max(maxField, match[1].length);
      maxType = Math.max(maxType, match[2].length);
    }
  }

  const formattedFields = fieldLines.map(line => {
    const trimmed = line.trim();
    const match = trimmed.match(/^(\w+)\s+([^\s\[\]]+)(.*)/);
    if (!match) return '  ' + trimmed;

    const [, field, type, rest] = match;
    const paddedField = field.padEnd(maxField + 1);
    const paddedType = type.padEnd(maxType + 1);
    return `  ${paddedField}${paddedType}${rest.trim()}`;
  });

  return [modelStart, ...formattedFields, modelEnd].join('\n');
}
