/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/
import { Test, TestingModule } from '@nestjs/testing';
import { DoctorService } from './doctor.service';
import { PrismaService } from '../prisma/prisma.service';
import { fuzzySearch } from 'nest-common-utilities';
import { ListDoctorsRequestDto } from './dto/list-doctors-request.dto';
import { AppException } from 'nest-common-utilities';

jest.mock('nest-common-utilities');

describe('DoctorService - Fuzzy Search', () => {
  let service: DoctorService;
  let mockPrismaService: any;
  let mockFuzzySearch: any;

  const mockDoctors = [
    {
      id: '1',
      name: 'Dr. Ashwin Shetty',
      registration_no: 'HR100926',
      doctor_speciality_master: { id: '1', name: 'Neurologist' },
      qualifications: [],
      pincode_master: { id: '1', name: '400011' },
      doctor_contact: '9876543210',
      doctor_email_id: 'ashwin@example.com',
      hpr_code: 'HPR123',
      pan_no: 'ABCDE1234F',
      address: '123 Street',
      latitude: '12.345',
      longitude: '78.901',
    },
    {
      id: '2',
      name: 'Dr. Ashwin Kumar',
      registration_no: 'HR100927',
      doctor_speciality_master: { id: '1', name: 'Neurologist' },
      qualifications: [],
      pincode_master: { id: '1', name: '400011' },
      doctor_contact: '9876543211',
      doctor_email_id: 'ashwin.k@example.com',
      hpr_code: 'HPR124',
      pan_no: 'ABCDE1235F',
      address: '124 Street',
      latitude: '12.346',
      longitude: '78.902',
    },
    {
      id: '3',
      name: 'Dr. John Doe',
      registration_no: 'HR100928',
      doctor_speciality_master: { id: '2', name: 'Cardiologist' },
      qualifications: [],
      pincode_master: { id: '2', name: '400012' },
      doctor_contact: '9876543212',
      doctor_email_id: 'john@example.com',
      hpr_code: 'HPR125',
      pan_no: 'ABCDE1236F',
      address: '125 Street',
      latitude: '12.347',
      longitude: '78.903',
    },
    {
      id: '4',
      name: 'Dr. Sarah Smith',
      registration_no: 'HR100929',
      doctor_speciality_master: { id: '3', name: 'Pediatrician' },
      qualifications: [],
      pincode_master: { id: '3', name: '400013' },
      doctor_contact: '9876543213',
      doctor_email_id: 'sarah@example.com',
      hpr_code: 'HPR126',
      pan_no: 'ABCDE1237F',
      address: '126 Street',
      latitude: '12.348',
      longitude: '78.904',
    },
  ];

  beforeEach(async () => {
    mockPrismaService = {
      doctor_master: {
        findMany: jest.fn(),
        count: jest.fn(),
      },
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        DoctorService,
        { provide: PrismaService, useValue: mockPrismaService },
      ],
    }).compile();

    service = module.get<DoctorService>(DoctorService);
    mockFuzzySearch = jest.mocked(fuzzySearch);
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  describe('getDoctors', () => {
    it('should return all doctors when no name is provided', async () => {
      const searchRequest = {
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      mockFuzzySearch.mockResolvedValue({
        data: mockDoctors,
        total: mockDoctors.length,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(4);
      expect(result.totalCount).toBe(4);
    });

    it('should return fuzzy search results for partial name match', async () => {
      const searchRequest = {
        doctorName: 'ash',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      const expectedDoctors = mockDoctors.filter((doctor) =>
        doctor.name.toLowerCase().includes('ash'),
      );

      mockFuzzySearch.mockResolvedValue({
        data: expectedDoctors,
        total: expectedDoctors.length,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(2);
      expect(result.totalCount).toBe(2);
      expect(result.data.every((doctor) => doctor.doctorName.toLowerCase().includes('ash'))).toBe(true);
    });

    it('should return empty array when no match is found', async () => {
      const searchRequest = {
        doctorName: 'xyz',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      mockFuzzySearch.mockResolvedValue({
        data: [],
        total: 0,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(0);
      expect(result.totalCount).toBe(0);
    });

    it('should handle case-insensitive search', async () => {
      const searchRequest = {
        doctorName: 'ASH',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      const expectedDoctors = mockDoctors.filter((doctor) =>
        doctor.name.toLowerCase().includes('ash'),
      );

      mockFuzzySearch.mockResolvedValue({
        data: expectedDoctors,
        total: expectedDoctors.length,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(2);
      expect(result.totalCount).toBe(2);
    });

    it('should filter by doctorRegNo', async () => {
      const searchRequest = {
        doctorRegNo: 'HR100926',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      const expectedDoctor = mockDoctors.find((doctor) =>
        doctor.registration_no === 'HR100926',
      );

      mockFuzzySearch.mockResolvedValue({
        data: [expectedDoctor],
        total: 1,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(1);
      expect(result.totalCount).toBe(1);
      expect(result.data[0].doctorRegNo).toBe('HR100926');
    });

    it('should filter by emailId', async () => {
      const searchRequest = {
        emailId: 'john@example.com',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      const expectedDoctor = mockDoctors.find((doctor) =>
        doctor.doctor_email_id === 'john@example.com',
      );

      mockFuzzySearch.mockResolvedValue({
        data: [expectedDoctor],
        total: 1,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(1);
      expect(result.totalCount).toBe(1);
      expect(result.data[0].emailId).toBe('john@example.com');
    });

    it('should filter by mobileNo', async () => {
      const searchRequest = {
        mobileNo: '9876543210',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      const expectedDoctor = mockDoctors.find((doctor) =>
        doctor.doctor_contact === '9876543210',
      );

      mockFuzzySearch.mockResolvedValue({
        data: [expectedDoctor],
        total: 1,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(1);
      expect(result.totalCount).toBe(1);
      expect(result.data[0].contactNo).toBe('9876543210');
    });

    it('should filter by hprCode', async () => {
      const searchRequest = {
        hprCode: 'HPR123',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      const expectedDoctor = mockDoctors.find((doctor) =>
        doctor.hpr_code === 'HPR123',
      );

      mockFuzzySearch.mockResolvedValue({
        data: [expectedDoctor],
        total: 1,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(1);
      expect(result.totalCount).toBe(1);
      expect(result.data[0].hprCode).toBe('HPR123');
    });

    it('should filter by panNo', async () => {
      const searchRequest = {
        panNo: 'ABCDE1234F',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      const expectedDoctor = mockDoctors.find((doctor) =>
        doctor.pan_no === 'ABCDE1234F',
      );

      mockFuzzySearch.mockResolvedValue({
        data: [expectedDoctor],
        total: 1,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(1);
      expect(result.totalCount).toBe(1);
      expect(result.data[0].panNo).toBe('ABCDE1234F');
    });

    it('should handle pagination with large page number', async () => {
      const searchRequest = {
        page: 100,
        limit: 10,
      } as ListDoctorsRequestDto;

      mockFuzzySearch.mockResolvedValue({
        data: [],
        total: mockDoctors.length,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(0);
      expect(result.totalCount).toBe(4);
    });

    it('should handle invalid limit value', async () => {
      const searchRequest = {
        page: 1,
        limit: 0,
      } as ListDoctorsRequestDto;

      mockFuzzySearch.mockResolvedValue({
        data: mockDoctors,
        total: mockDoctors.length,
      });

      const result = await service.getDoctors(searchRequest);
      expect(result.data).toHaveLength(4); // returns all items when invalid limit
      expect(result.totalCount).toBe(4);
    });

    it('should handle error from fuzzySearch', async () => {
      const searchRequest = {
        doctorName: 'test',
        page: 1,
        limit: 10,
      } as ListDoctorsRequestDto;

      mockFuzzySearch.mockRejectedValue(new AppException('Database error', 500));

      // The service catches the error and doesn't throw it, so we expect it to return undefined
      const result = await service.getDoctors(searchRequest);
      expect(result).toBeUndefined();
    });
  });
});
