/**
 * @file        microservice.service.ts
 * @description Service for checking the availability of microservices using NestJS ClientProxy communication pattern.
 * @author      <Your Name>
 * @created     <YYYY-MM-DD>
 **/

import { Injectable } from '@nestjs/common';
import { ModuleRef } from '@nestjs/core';
import { ClientProxy } from '@nestjs/microservices';
import { firstValueFrom } from 'rxjs';
import { Logger } from 'nest-common-utilities';

/**
 * MicroserviceHealthService provides methods to check the health of registered microservices.
 */
@Injectable()
export class MicroserviceHealthService {
    private logger = new Logger(MicroserviceHealthService.name);

    /**
     * Initializes the MicroserviceHealthService.
     * 
     * @param moduleRef - Reference to the application module used to resolve dynamic providers.
     */
    constructor(
        private moduleRef: ModuleRef,
    ) { }

    /**
     * Checks if a microservice registered with the given token is available by sending a 'ping' command.
     * 
     * @param token - The DI token used to register the microservice (e.g., 'AUTH_SERVICE').
     * @returns A promise that resolves to true if the microservice responds with 'pong', otherwise false.
     */
    async isServiceAvailable(token: string): Promise<boolean> {
        try {
            const client: ClientProxy = this.moduleRef.get<ClientProxy>(token, { strict: false });
            const result = await client.send({ cmd: 'ping' }, {});
            const response = await firstValueFrom(result);

            this.logger.info(`Response from microservice "${token}": ${response}`);
            return response === 'pong';
        } catch (error) {
            this.logger.error(`Failed to contact microservice "${token}": ${error.message}`);
            return false;
        }
    }

    /**
     * Checks the availability of multiple microservices.
     * 
     * @param tokens - An array of DI tokens representing the microservices to be checked.
     * @returns A record mapping each token to a boolean indicating its availability.
     */
    async checkMultipleServices(tokens: string[]): Promise<Record<string, boolean>> {
        const statuses: Record<string, boolean> = {};
        for (const token of tokens) {
            statuses[token] = await this.isServiceAvailable(token);
        }
        return statuses;
    }
}
