/**
 @file        <file name>
 @description  <description>
 @author      <Your Name>
 @created     <YYYY-MM-DD>
**/

import { NestFactory, Reflector } from '@nestjs/core';
import { ClassSerializerInterceptor, ValidationPipe } from '@nestjs/common';
import { AppModule } from './app.module';
import helmet from 'helmet';
import { setupSwagger } from './config/swagger.config';
import { Logger } from 'nest-common-utilities';
import {
  GlobalExceptionFilter,
  ResponseInterceptor,
  DecryptMiddleware,
  EncryptInterceptor,
  setPiiFields
} from 'nest-common-utilities';
import { MicroserviceOptions, Transport } from '@nestjs/microservices';
import { config } from './config/pii.config'

/**
 * The main function that bootstraps the NestJS application, sets up global middleware,
 * exception filters, interceptors, CORS, security headers, and Swagger documentation.
 * 
 * @returns {Promise<void>} A promise that resolves once the application has started.
 */
async function bootstrap(): Promise<void> {
  const app = await NestFactory.create(AppModule);
  const logger = new Logger('app');

  setPiiFields(config);

  // Global filters and interceptors
  app.useGlobalFilters(new GlobalExceptionFilter());
  app.useGlobalInterceptors(new ClassSerializerInterceptor(app.get(Reflector)));
  app.useGlobalInterceptors(new ResponseInterceptor());

  app.useGlobalPipes(
    new ValidationPipe({
      transform: true, whitelist: true, forbidNonWhitelisted: true,
    }),
  );

  // Security and CORS
  app.use(helmet());
  app.enableCors();

  // Swagger documentation (enabled in non-prod environments)
  if (process.env.NODE_ENV !== 'prod') {
    setupSwagger(app);
  }

  // Microservice setup using TCP transport
  const microservice = await NestFactory.createMicroservice<MicroserviceOptions>(AppModule, {
    transport: Transport.TCP,
    options: {
      host: '0.0.0.0',
      port: 8002,
    },
  });

  // Global filters and interceptors for microservice
  microservice.useGlobalFilters(new GlobalExceptionFilter());
  microservice.useGlobalInterceptors(new ClassSerializerInterceptor(microservice.get(Reflector)));
  microservice.useGlobalInterceptors(new ResponseInterceptor());
  microservice.useGlobalPipes(
    new ValidationPipe({
      transform: true, whitelist: true, forbidNonWhitelisted: true,
    }),
  );

  await microservice.listen();
  logger.info(`Microservice started on port: 8002`);

  // Start HTTP server
  const port = process.env.PORT || 3002;
  await app.listen(port, () => {
    logger.info(`Server started on port: ${port}`);
  });
}

// Start the app
bootstrap();
